<?php
// vim: set ts=4 sw=4 sts=4 et:

/**
 * X-Cart
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the software license agreement
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.x-cart.com/license-agreement.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to licensing@x-cart.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not modify this file if you wish to upgrade X-Cart to newer versions
 * in the future. If you wish to customize X-Cart for your needs please
 * refer to http://www.x-cart.com/ for more information.
 *
 * @category  X-Cart 5
 * @author    Magic Toolbox <support@magictoolbox.com>
 * @copyright Copyright (c) 2013 Magic Toolbox <support@magictoolbox.com>. All rights reserved
 * @license   http://www.x-cart.com/license-agreement.html X-Cart 5 License Agreement
 * @link      http://www.magictoolbox.com/
*/

namespace XLite\Module\MagicToolbox\MagicZoomPlus\Controller\Admin;

/**
 * MagicZoomPlus settings page controller
 *
 */
class MagiczoomplusSettings extends \XLite\Controller\Admin\AAdmin
{
    /**
     * Pages
     */
    const PAGE_DEFAULTS                    = 'default';
    const PAGE_PRODUCT_SETTINGS            = 'product';
    const PAGE_RESET_SETTINGS              = 'reset';

    /**
     * Controller parameters
     *
     * @var array
     */
    protected $params = array('target', 'page');

    /**
     * Return the current page title (for the content area)
     *
     * @return string
     */
    public function getTitle()
    {
        return static::t('Magic Zoom Plus module settings');
    }

    /**
     * Handles the request.
     *
     * @return void
     */
    public function handleRequest()
    {
        parent::handleRequest();

        $repo = \XLite\Core\Database::getRepo('\XLite\Module\MagicToolbox\MagicZoomPlus\Model\Config');
        $doLoadDefaults = !($repo && $repo->count());

        if ($doLoadDefaults || \XLite\Core\Request::getInstance()->page == static::PAGE_RESET_SETTINGS) {
            $this->doLoadDefaults();
        }

        if (!$this->isPageValid(\XLite\Core\Request::getInstance()->page)) {
            $this->setHardRedirect();
            $this->setReturnURL(
                $this->buildURL(
                    'magiczoomplus_settings',
                    '',
                    array(
                        'page' => $this->getDefaultPage(),
                    )
                )
            );
            $this->doRedirect();
        }
    }

    /**
     * Check if page is valid
     *
     * @param string $page Page to check
     *
     * @return boolean
     */
    public function isPageValid($page)
    {
        return in_array(strval($page), array_keys(self::getAllPages()));
    }

    /**
     * Get all pages
     *
     * @return string
     */
    public static function getAllPages()
    {
        return array(
            static::PAGE_DEFAULTS                    => static::t('Defaults'),
            static::PAGE_PRODUCT_SETTINGS            => static::t('Product page'),
        );
    }

    /**
     * Get default page
     *
     * @return string
     */
    public function getDefaultPage()
    {
        return static::PAGE_DEFAULTS;
    }

    /**
     * Get pages
     *
     * @return array
     */
    public function getPages()
    {
        $list = parent::getPages();
        $list += $this->getAllPages();
        return $list;
    }

    /**
     * Get pages templates
     *
     * @return array
     */
    protected function getPageTemplates()
    {
        $list = parent::getPageTemplates();
        foreach ($this->getAllPages() as $page => $title) {
            $list[$page] = 'modules/MagicToolbox/MagicZoomPlus/settings.twig';
        }
        return $list;
    }

    /**
     * Get get settings container additional class
     *
     * @return string
     */
    public function getSettingsContainerAdditionalClass()
    {
        $class = '';
        $version = \XLite::getInstance()->getVersion();
        if (version_compare($version, '5.4', '>=')) {
            $class = 'xlite-54x';
        }
        return $class;
    }

    /**
     * Get class name for the form
     *
     * @return string
     */
    protected function getModelFormClass()
    {
        return '\XLite\Module\MagicToolbox\MagicZoomPlus\View\Model\Settings';
    }

    /**
     * Update module settings
     *
     * @return void
     */
    public function doActionUpdate()
    {
        $page = \XLite\Core\Request::getInstance()->page;

        if ($page == static::PAGE_DEFAULTS) {
            $licenseKeys = array();
            $options = \XLite\Core\Database::getRepo('\XLite\Module\MagicToolbox\MagicZoomPlus\Model\Config')->findByProfileAndNames(
                $page,
                array('license-key', 'magicscroll-license-key')
            );
            foreach ($options as $key => $option) {
                $licenseKeys[$option->name] = $option->value;
            }
        }

        $this->getModelForm()->performAction('update');

        if ($page == static::PAGE_DEFAULTS) {
            $options = \XLite\Core\Database::getRepo('\XLite\Module\MagicToolbox\MagicZoomPlus\Model\Config')->findByProfileAndNames(
                $page,
                array('license-key', 'magicscroll-license-key')
            );
            foreach ($options as $key => $option) {
                if ($option->value != $licenseKeys[$option->name]) {
                    if (empty($option->value) || preg_match('#[^\-\.0-9A-Za-z]#', $option->value)) {
                        \XLite\Core\TopMessage::addWarning('Please enter the correct license key.');
                    } else {
                        if ($option->name == 'license-key') {
                            $this->doUpdateLicense($option->value, 'magiczoomplus');
                        } elseif ($option->name == 'magicscroll-license-key') {
                            $this->doUpdateLicense($option->value, 'magicscroll');
                        }
                    }
                }
            }
        }
    }

    /**
     * Update license
     *
     * @return void
     */
    protected function doUpdateLicense($licenseKey, $fileName)
    {
        $url = "https://www.magictoolbox.com/site/order/{$licenseKey}/{$fileName}.js";
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_BINARYTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'GET');
        $response = curl_exec($ch);
        $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        if ($code == 200) {
            $jsPath = "modules/MagicToolbox/MagicZoomPlus/js/{$fileName}.js";
            $jsPath = \XLite\Core\Layout::getInstance()->getResourceFullPath($jsPath, \XLite::CUSTOMER_INTERFACE, false);
            if (\Includes\Utils\FileManager::write($jsPath, $response, 0, 0755)) {
                \XLite\Core\TopMessage::addInfo('License have been updated successfully.');
                $href = \XLite::ADMIN_SELF.'?'.http_build_query(array('target' => 'css_js_performance'));
                $href = \XLite\Core\URLManager::getShopURL($href);
                \XLite\Core\TopMessage::addInfo("Please don't forget to <a href=\"{$href}\">clear aggregation cache</a> if need it.");
            }
        } elseif ($code == 403) {
            \XLite\Core\TopMessage::addWarning('There was a problem with checking your license key. Please <a target="_blank" href="http://www.magictoolbox.com/contact/">contact us</a>.');
        } else {
            \XLite\Core\TopMessage::addWarning('Please enter the valid license key.');
        }
    }

    /**
     * Load defaults
     *
     * @return void
     */
    protected function doLoadDefaults()
    {
        $yaml = \Includes\Utils\ModulesManager::getModuleYAMLFiles('MagicToolbox', 'MagicZoomPlus');
        $yaml = $yaml[0];
        if (\Includes\Utils\FileManager::isFileReadable($yaml)) {
            \XLite\Core\Database::getRepo('\XLite\Module\MagicToolbox\MagicZoomPlus\Model\Config')->clearAll();
            \XLite\Core\Database::getInstance()->loadFixturesFromYaml($yaml);
        }
    }

    /**
     * Return module options
     *
     * @return array
     */
    public function getOptions()
    {
        $profile = \XLite\Core\Request::getInstance()->page;
        return \XLite\Core\Database::getRepo('\XLite\Module\MagicToolbox\MagicZoomPlus\Model\Config')->findByProfile($profile);
    }
}
