<?php
// vim: set ts=4 sw=4 sts=4 et:

/**
 * X-Cart
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the software license agreement
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.x-cart.com/license-agreement.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to licensing@x-cart.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not modify this file if you wish to upgrade X-Cart to newer versions
 * in the future. If you wish to customize X-Cart for your needs please
 * refer to http://www.x-cart.com/ for more information.
 *
 * @category  X-Cart 5
 * @author    Magic Toolbox <support@magictoolbox.com>
 * @copyright Copyright (c) 2013 Magic Toolbox <support@magictoolbox.com>. All rights reserved
 * @license   http://www.x-cart.com/license-agreement.html X-Cart 5 License Agreement
 * @link      http://www.magictoolbox.com/
*/

namespace XLite\Module\MagicToolbox\MagicZoom\View\Product\Details\Customer;

/**
 * MagicZoom
 *
 */
class MagicZoom extends \XLite\View\Product\Details\Customer\ACustomer
{
    /**
     * Product unique ID
     *
     * @var   integer
     *
     */
    protected $currentProductId = null;

    /**
     * Main Image HTML
     *
     * @var   string
     */
    protected $mainImageHTML = '';

    /**
     * Selectors HTML array
     *
     * @var   array
     */
    protected $selectorsHTML = array();

    /**
     * Current layout
     *
     * @var   string
     */
    protected $magicToolboxLayout = 'original';

    /**
     * Additional classes
     *
     * @var   string
     *
     */
    protected $additionalClasses = '';

    /**
     * Scroll options
     *
     * @var   string
     *
     */
    protected $scrollOptions = '';

    /**
     * Get JS files list
     *
     * @return array
     */
    public function getJSFiles()
    {
        $list = parent::getJSFiles();

        $tool = \XLite\Module\MagicToolbox\MagicZoom\Classes\Helper::getInstance()->getPrimaryTool();
        if ($tool->params->checkValue('enable-effect', 'Yes', 'product')) {
            $list[] = 'modules/MagicToolbox/MagicZoom/js/controller.js';
        }

        return $list;
    }

    /**
     * Return widget default template
     *
     * @return string
     */
    protected function getDefaultTemplate()
    {
        return 'modules/MagicToolbox/MagicZoom/templates/magiczoom.twig';
    }

    /**
     * Check visibility
     *
     * @return boolean
     */
    protected function isVisible()
    {
        return parent::isVisible();
    }

    /**
     * Check whether to display gallery as hidden
     *
     * @return boolean
     */
    protected function isVisibleAsHidden()
    {
        return count($this->selectorsHTML) < 2;
    }

    /**
     * Method to get main image HTML
     *
     * @return string
     */
    public function getMainImageHTML()
    {
        return $this->mainImageHTML;
    }

    /**
     * Method to get selectors HTML
     *
     * @return array
     */
    public function getSelectorsHTML()
    {
        return $this->selectorsHTML;
    }

    /**
     * Method to get selectors count
     *
     * @return integer
     */
    public function getSelectorsCount()
    {
        return count($this->selectorsHTML);
    }

    /**
     * Method to get current layout
     *
     * @return string
     */
    public function getLayout()
    {
        return $this->magicToolboxLayout;
    }

    /**
     * Method to get ID
     *
     * @return integer|string
     */
    public function getPid()
    {
        return ($this->currentProductId !== null) ? $this->currentProductId : '';
    }

    /**
     * Method to get additional classes
     *
     * @return string
     */
    public function getAdditionalClasses()
    {
        return $this->additionalClasses;
    }

    /**
     * Method to get scroll options
     *
     * @return string
     */
    public function getScrollOptions()
    {
        return $this->scrollOptions;
    }

    /**
     * Method to get option value
     *
     * @param string $option Option name
     *
     * @return mixed
     */
    public function getOptionValue($option)
    {
        $tool = \XLite\Module\MagicToolbox\MagicZoom\Classes\Helper::getInstance()->getPrimaryTool();
        $tool->params->setProfile('product');
        return $tool->params->getValue($option);
    }

    /**
     * Method to render MagicZoom HTML
     *
     * @return boolean
     */
    public function renderTemplate()
    {
        $helper = \XLite\Module\MagicToolbox\MagicZoom\Classes\Helper::getInstance();
        $tool = $helper->getPrimaryTool();
        $tool->params->setProfile('product');
        if ($tool->params->checkValue('enable-effect', 'No')) {
            return false;
        }

        $thumbMaxWidth = intval($tool->params->getValue('thumb-max-width', 'product'));
        $thumbMaxHeight = intval($tool->params->getValue('thumb-max-height', 'product'));
        $selectorMaxWidth = intval($tool->params->getValue('selector-max-width', 'product'));
        $selectorMaxHeight = intval($tool->params->getValue('selector-max-height', 'product'));
        $useAlt = $tool->params->checkValue('use-individual-titles', 'Yes', 'product');

        $product = $this->getProduct();
        $this->currentProductId = $product->getId();
        $productName = $product->getName();
        $briefDescription = $product->getBriefDescription();
        $fullDescription = $product->getDescription();
        $images = $product->getPublicImages();
        $imagesCount = count($images);
        if (!$imagesCount) {
            return false;
        }

        foreach ($images as $index => $image) {

            $img = $image->getURL();
            list($width, $height, $thumb) = $image->doResize($thumbMaxWidth, $thumbMaxHeight, false);
            list($width, $height, $selector) = $image->doResize($selectorMaxWidth, $selectorMaxHeight, false);
            $alt = $useAlt ? $image->getAlt() : '';

            if ($index == 0) {
                $this->mainImageHTML = $tool->getMainTemplate(array(
                    'id' => 'Product' . $this->currentProductId,
                    'img' => $img,
                    'thumb' => $thumb,
                    'title' => $alt ? $alt : $productName,
                    'briefDescription' => $briefDescription,
                    'fullDescription' => $fullDescription,
                ));
            }
            $selectorsHTML = $tool->getSelectorTemplate(array(
                'id' => 'Product' . $this->currentProductId,
                'group' => 'productPage',
                'img' => $img,
                'medium' => $thumb,
                'thumb' => $selector,
                'title' => $alt ? $alt : $productName
            ));
            if ($image instanceof \XLite\Module\XC\ProductVariants\Model\Image\ProductVariant\Image) {
                $selectorsHTML = preg_replace('#^<a #', '<a data-variant-image="true" ', $selectorsHTML);
            }
            $this->selectorsHTML[] = $selectorsHTML;

        }

        $this->magicToolboxLayout = $tool->params->getValue('template');

        if ($tool->params->checkValue('magicscroll', 'Yes')) {
            $additionalClasses = $tool->params->getValue('scroll-extra-styles');
            if (empty($additionalClasses)) {
                $this->additionalClasses = ' MagicScroll';
            } else {
                $this->additionalClasses = ' MagicScroll ' . trim($additionalClasses);
            }
            $scroll = $helper->getScrollTool();
            $dataOptions = $scroll->params->serialize(false, '', 'product-magicscroll-options');
            if (empty($dataOptions)) {
                $this->scrollOptions = '';
            } else {
                $this->scrollOptions = " data-options=\"{$dataOptions}\"";
            }
        }

        return true;
    }

    /**
     * Method to get default image URL
     *
     * @return string
     */
    public function getDefaultImageURL()
    {
        $url = null;

        $url = \XLite::getInstance()->getOptions(['images', 'default_image']);

        if (!\XLite\Core\Converter::isURL($url)) {
            $url = \XLite\Core\Layout::getInstance()->getResourceWebPath(
                $url,
                \XLite\Core\Layout::WEB_PATH_OUTPUT_URL
            );
        }

        $url = str_replace(array('http://', 'https://'), '//', $url);

        return $url;
    }

    /**
     * Method to get default image HTML
     *
     * @return string
     */
    public function getDefaultImageHTML()
    {
        $product = $this->getProduct();
        $id = $product->getId();
        $productName = $product->getName();
        $img = $this->getDefaultImageURL();
        $tool = \XLite\Module\MagicToolbox\MagicZoom\Classes\Helper::getInstance()->getPrimaryTool();
        $briefDescription = $product->getBriefDescription();
        $fullDescription = $product->getDescription();
        $html = $tool->getMainTemplate(array(
            'id' => 'Product' . $id,
            'img' => $img,
            'thumb' => $img,
            'title' => $productName,
            'briefDescription' => $briefDescription,
            'fullDescription' => $fullDescription,
        ));
        return $html;
    }
}
