<?php
// vim: set ts=4 sw=4 sts=4 et:

/**
 * X-Cart
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the software license agreement
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.x-cart.com/license-agreement.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to licensing@x-cart.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not modify this file if you wish to upgrade X-Cart to newer versions
 * in the future. If you wish to customize X-Cart for your needs please
 * refer to http://www.x-cart.com/ for more information.
 *
 * @category  X-Cart 5
 * @author    Magic Toolbox <support@magictoolbox.com>
 * @copyright Copyright (c) 2013 Magic Toolbox <support@magictoolbox.com>. All rights reserved
 * @license   http://www.x-cart.com/license-agreement.html X-Cart 5 License Agreement
 * @link      http://www.magictoolbox.com/
*/

namespace XLite\Module\MagicToolbox\MagicThumb\View\Model;

/**
 * Settings dialog model widget
 */
class Settings extends \XLite\View\Model\AModel
{
    /**
     * Indexes in field schemas
     *
     */
    const SCHEMA_STATUS = 'paramStatus';

    /**
     * Get schema fields
     *
     * @return array
     */
    public function getSchemaFields()
    {
        $list = array();

        foreach ($this->getOptions() as $option) {
            $cell = $this->getFormFieldByOption($option);
            if ($cell) {
                $list[$option->getName()] = $cell;
            }
        }

        return $list;
    }

    /**
     * Get form field by option
     *
     * @param \XLite\Module\MagicToolbox\MagicThumb\Model\Config $option Option
     *
     * @return array
     */
    protected function getFormFieldByOption(\XLite\Module\MagicToolbox\MagicThumb\Model\Config $option)
    {
        $cell = null;

        $class = $this->detectFormFieldClassByOption($option);

        if ($class) {
            $cell = array(
                self::SCHEMA_CLASS    => $class,
                self::SCHEMA_LABEL    => $option->getOptionName(),
                self::SCHEMA_HELP     => $option->getOptionComment(),
                self::SCHEMA_REQUIRED => false,
                self::SCHEMA_STATUS   => $option->getStatus(),
            );

            if ($this->isOptionRequired($option)) {
                $cell[self::SCHEMA_REQUIRED] = true;
            }

            $parameters = $option->getWidgetParameters();
            if ($parameters && is_array($parameters)) {
                $cell += $parameters;
            }
        }

        return $cell;
    }

    /**
     * Detect form field class by option
     *
     * @param \XLite\Module\MagicToolbox\MagicThumb\Model\Config $option Option
     *
     * @return string
     */
    protected function detectFormFieldClassByOption(\XLite\Module\MagicToolbox\MagicThumb\Model\Config $option)
    {
        $class = null;
        $type = $option->getType() ?: 'text';

        switch ($type) {
            case 'textarea':
                $class = '\XLite\View\FormField\Textarea\Simple';
                break;
            case 'checkbox':
                $class = '\XLite\View\FormField\Input\Checkbox\Simple';
                break;
            case 'country':
                $class = '\XLite\View\FormField\Select\Country';
                break;
            case 'state':
                $class = '\XLite\View\FormField\Select\State';
                break;
            case 'currency':
                $class = '\XLite\View\FormField\Select\Currency';
                break;
            case 'separator':
                $class = '\XLite\View\FormField\Separator\Regular';
                break;
            case 'text':
                $class = '\XLite\View\FormField\Input\Text';
                break;
            case 'hidden':
                break;
            default:
                if (preg_match('/^\\\?XLite\\\/Ss', $option->getType())) {
                    $class = $option->getType();
                }
        }

        return $class;
    }

    /**
     * Check - option is required or not
     *
     * @param \XLite\Module\MagicToolbox\MagicThumb\Model\Config $option Option
     *
     * @return boolean
     */
    protected function isOptionRequired(\XLite\Module\MagicToolbox\MagicThumb\Model\Config $option)
    {
        return false;
    }

    /**
     * Get form fields for default section
     *
     * @return array
     */
    protected function getFormFieldsForSectionDefault()
    {
        $result = $this->getFieldsBySchema($this->getSchemaFields());
        return $result;
    }

    /**
     * Return list of the "Button" widgets
     *
     * @return array
     */
    protected function getFormButtons()
    {
        $result = array();

        $result['submit'] = new \XLite\View\Button\Submit(
            array(
                \XLite\View\Button\AButton::PARAM_LABEL    => 'Save settings',
                \XLite\View\Button\AButton::PARAM_BTN_TYPE => 'regular-main-button',
                \XLite\View\Button\AButton::PARAM_STYLE    => 'action',
            )
        );

        $version = \XLite::getInstance()->getVersion();
        if (version_compare($version, '5.4', '<')) {
            $moduleID = $this->getModuleId();

            $result['addons-list'] = new \XLite\View\Button\BackToModulesLink(
                array(
                    \XLite\View\Button\BackToModulesLink::PARAM_MODULE_ID => $moduleID,
                    \XLite\View\Button\AButton::PARAM_STYLE               => 'action addons-list-back-button',
                )
            );
        }

        return $result;
    }

    /**
     * Gets the module ID
     *
     * @return integer
     */
    public function getModuleId()
    {
        static $moduleID = null;
        if ($moduleID === null) {
            $module = \XLite\Core\Database::getRepo('XLite\Model\Module')->findOneBy(array(
                'author'            => 'MagicToolbox',
                'name'              => 'MagicThumb',
                'fromMarketplace'   => false,
            ));
            if ($module) {
                $moduleID = $module->getModuleID();
            }
        }
        return $moduleID;
    }

    /**
     * Perform certain action for the model object
     *
     * @return boolean
     */
    protected function performActionUpdate()
    {
        return true;
    }

    /**
     * Retrieve property from the model object
     *
     * @param mixed $name Field/property name
     *
     * @return mixed
     */
    protected function getModelObjectValue($name)
    {
        $value = null;

        foreach ($this->getOptions() as $option) {
            if ($option->getName() == $name) {
                $value = $option->getValue();
                break;
            }
        }

        return $value;
    }

    /**
     * Populate model object properties by the passed data
     *
     * @param array $data Data to set
     *
     * @return void
     */
    protected function setModelProperties(array $data)
    {
        $optionsToUpdate = array();

        //NOTE: find changed options and store them in $optionsToUpdate
        foreach ($this->getEditableOptions() as $key => $option) {
            $name  = $option->name;
            $type  = $option->type;

            if ('checkbox' == $type) {
                $newValue = empty($data[$name]) ? 'N' : 'Y';
            } elseif ('serialized' == $type && isset($data[$name]) && is_array($data[$name])) {
                $newValue = serialize($data[$name]);
            } elseif ('text' == $type) {
                $newValue = isset($data[$name]) ? trim($data[$name]) : '';
            } elseif ('XLite\Module\MagicToolbox\MagicThumb\View\FormField\MagicToolboxLicenseText' == $type) {
                $newValue = isset($data[$name]) ? strtoupper(trim($data[$name])) : '';
            } else {
                $newValue = isset($data[$name]) ? $data[$name] : '';
            }

            if (isset($data[$name])) {
                if ($option->value != $newValue) {
                    $option->value = $newValue;
                    if ($option->status == 0) {
                        $option->status = 1;
                    }
                    $optionsToUpdate[] = $option;

                    //NOTE: change appropriate inactive options when saving default
                    if ('default' == $option->profile) {
                        $foundOptions = \XLite\Core\Database::getRepo('\XLite\Module\MagicToolbox\MagicThumb\Model\Config')->findBy(array(
                            'name' => $option->name,
                            'status' => 0,
                        ));
                        foreach ($foundOptions as $foundOption) {
                            if ($foundOption->value != $option->value) {
                                $foundOption->value = $option->value;
                                $optionsToUpdate[] = $foundOption;
                            }
                        }
                    }

                } elseif ($option->status == 0) {
                    $option->status = 1;
                    $optionsToUpdate[] = $option;
                }
            } elseif ($option->status == 1) {
                $option->status = 0;

                //NOTE: set value to default if exists
                $default = \XLite\Core\Database::getRepo('\XLite\Module\MagicToolbox\MagicThumb\Model\Config')->findOneBy(array(
                    'profile' => 'default',
                    'name'    => $option->name,
                ));
                if ($default) {
                    $option->value = $default->value;
                }

                $optionsToUpdate[] = $option;
            }

        }

        //NOTE: save changed options to the database
        if (!empty($optionsToUpdate)) {
            \XLite\Core\Database::getEM()->flush();
        }
    }

    /**
     * Allow tags for some parameters
     *
     * @param string $name Param name
     *
     * @return boolean
     */
    protected function isParamAllowsTags($name)
    {
        $list = array('images', 'message');
        return parent::isParamAllowsTags($name) || in_array($name, $list);
    }

    /**
     * Return true if param value may contain anything
     *
     * @param string $name Param name
     *
     * @return boolean
     */
    protected function isParamTrusted($name)
    {
        $list = array('images', 'message');
        return parent::isParamTrusted($name) || in_array($name, $list);
    }

    /**
     * Get editable options
     *
     * @return array
     */
    protected function getEditableOptions()
    {
        $options = $this->getOptions();
        $exclude = array('separator', 'hidden');
        foreach ($options as $key => $option) {
            if (in_array($option->type, $exclude)) {
                unset($options[$key]);
            }
        }
        return $options;
    }

    /**
     * This object will be used if another one is not passed
     *
     * @return \XLite\Model\Profile
     */
    protected function getDefaultModelObject()
    {
        return null;
    }

    /**
     * Return name of web form widget class
     *
     * @return string
     */
    protected function getFormClass()
    {
        return '\XLite\Module\MagicToolbox\MagicThumb\View\Form\Settings';
    }

    protected function addDataSavedTopMessage()
    {
        \XLite\Core\TopMessage::addInfo('Magic Thumb module settings have been saved successfully');
    }
}
