<?php
// vim: set ts=4 sw=4 sts=4 et:

/**
 * X-Cart
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the software license agreement
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.x-cart.com/license-agreement.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to licensing@x-cart.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not modify this file if you wish to upgrade X-Cart to newer versions
 * in the future. If you wish to customize X-Cart for your needs please
 * refer to http://www.x-cart.com/ for more information.
 *
 * @category  X-Cart 5
 * @author    Magic Toolbox <support@magictoolbox.com>
 * @copyright Copyright (c) 2013 Magic Toolbox <support@magictoolbox.com>. All rights reserved
 * @license   http://www.x-cart.com/license-agreement.html X-Cart 5 License Agreement
 * @link      http://www.magictoolbox.com/
*/

namespace XLite\Module\MagicToolbox\MagicScroll\Model\Repo;

/**
 * MagicScroll configuration registry
 */
class Config extends \XLite\Model\Repo\Base\I18n
{
    /**
     * Repository type
     *
     * @var string
     */
    protected $type = self::TYPE_SERVICE;

    /**
     * Default 'order by' field name
     *
     * @var string
     */
    protected $defaultOrderBy = 'orderby';

    /**
     * Alternative record identifiers
     *
     * @var array
     */
    protected $alternativeIdentifier = array(
        array('profile', 'name'),
    );

    /**
     * Get the list of options of the specified profile
     *
     * @param string  $profile      Profile
     * @param boolean $force        Force OPTIONAL
     * @param boolean $doNotProcess Do not process OPTIONAL
     *
     * @return array
     */
    public function getByProfile($profile, $force = false, $doNotProcess = false)
    {
        $data = null;
        if (!$force) {
            $data = $this->getFromCache('profile', array('profile' => $profile));
        }
        if (!isset($data)) {
            $data = $this->findBy(array('profile' => $profile), array('orderby' => 'asc'));
            if (!$doNotProcess) {
                $data = $this->processOptions($data);
                $this->saveToCache($data, 'profile', array('profile' => $profile));
            }
        }
        return $data;
    }

    /**
     * Find all settings by profile name
     *
     * @param string $profile Profile name
     *
     * @return array
     */
    public function findByProfile($profile)
    {
        $result = $this->getByProfile($profile, true, true);
        if ($result) {
            foreach ($result as $k => $v) {
                if (empty($v->type)) {
                    unset($result[$k]);
                }
            }
        }
        return $result ?: array();
    }

    /**
     * Find settings by profile name and option names
     *
     * @param string $profile Profile name
     * @param array $names Array of option names
     *
     * @return array
     */
    public function findByProfileAndNames($profile, $names = array())
    {
        $result = $this->getByProfile($profile, true, true);
        if ($result) {
            foreach ($result as $k => $v) {
                if (!in_array($v->name, $names)) {
                    unset($result[$k]);
                }
            }
        }
        return $result ?: array();
    }


    /**
     * Get the list of all options
     *
     * @param boolean $force Do not use cache OPTIONAL
     *
     * @return array
     */
    public function getAllOptions($force = false)
    {
        $data = null;
        if (!$force) {
            $data = $this->getFromCache('all');
        }
        if (!isset($data)) {
            $data = $this->createQueryBuilder()->getResult();
            $data = $this->detachList($data);
            $data = $this->processOptions($data);
            $this->saveToCache($data, 'all');
        }
        return $data;
    }

    /**
     * Get the list of editable options
     *
     * @param boolean $force Do not use cache OPTIONAL
     *
     * @return array
     */
    public function getEditableAndActiveOptions($force = false)
    {
        $data = null;
        if (!$force) {
            $data = $this->getFromCache('editable');
        }
        if (!isset($data)) {
            $data = $this->createQueryBuilder()->getResult();
            $data = $this->detachList($data);
            $exclude = array('separator', 'hidden');
            foreach ($data as $key => $option) {
                if ($option->getStatus() == \XLite\Module\MagicToolbox\MagicScroll\Model\Config::OPTION_IS_INACTIVE || in_array($option->getType(), $exclude)) {
                    unset($data[$key]);
                }
            }
            $data = $this->processOptions($data);
            $this->saveToCache($data, 'editable');
        }
        return $data;
    }

    /**
     * Preprocess options and transform its to the hierarchy of \XLite\Core\ConfigCell objects
     *
     * @param array $data Array of options data gathered from the database
     *
     * @return \XLite\Core\ConfigCell
     */
    public function processOptions($data)
    {
        $config = new \XLite\Core\ConfigCell();

        foreach ($data as $option) {

            $profile = $option->getProfile();
            $name    = $option->getName();
            $value   = $option->getValue();

            if (!isset($config->$profile)) {
                $config->$profile = new \XLite\Core\ConfigCell();
            }

            $config->$profile->$name = $value;

        }

        return $config;
    }

    /**
     * Create new option / Update option value
     *
     * @param array $data Option data in the following format
     *
     * @return void
     * @throws \Exception
     */
    public function createOption($data)
    {
        // Array of allowed fields and flag required/optional
        $fields = array(
            'profile' => 1,
            'name'    => 1,
            'value'   => 1,
            'type'    => 0,
            'orderby' => 0,
            'default_value' => 0,
            'status' => 0,
        );

        $errorFields = array();

        foreach ($fields as $field => $required) {
            if (isset($data[$field])) {
                $fields[$field] = $data[$field];
            } elseif ($required) {
                $errorFields[] = $field;
            }
        }

        if (!empty($errorFields)) {
            throw new \Exception(
                'createOptions() failed: The following required fields are missed: '
                . implode(', ', $errorFields)
            );
        }

        if (isset($fields['type']) && !$this->isValidOptionType($fields['type'])) {
            throw new \Exception('createOptions() failed: Wrong option type: ' . $type);
        }

        $option = $this->findOneBy(array('name' => $fields['name'], 'profile' => $fields['profile']));

        if ($option) {
            // Existing option
            $option->setValue($fields['value']);
        } else {
            // Create a new option
            $option = new \XLite\Module\MagicToolbox\MagicScroll\Model\Config();
            $option->map($fields);
            \XLite\Core\Database::getEM()->persist($option);
        }

        \XLite\Core\Database::getEM()->flush();

    }

    /**
     * Define cache cells
     *
     * @return array
     */
    protected function defineCacheCells()
    {
        $list = parent::defineCacheCells();
        $list['all'] = array();
        $list['profile'] = array(
            self::ATTRS_CACHE_CELL => array('profile')
        );
        $list['editable'] = array();
        return $list;
    }

    /**
     * Check if option type is a valid
     *
     * @param string $optionType Option type
     *
     * @return boolean
     */
    protected function isValidOptionType($optionType)
    {
        $simple = in_array(
            $optionType,
            array(
                '',
                'text',
                'radio',
                'select',
                'separator',
                'textarea',
                'checkbox',
                'serialized',
                'hidden'
            )
        );

        if (!$simple && preg_match('/^XLite\\\(Module\\\.+\\\)?View\\\FormField\\\/Ss', $optionType)) {
            $simple = true;
        }

        return $simple;
    }
}
