<?php

/**
 * Magic Thumb view block
 *
 */
namespace MagicToolbox\MagicThumb\Block\Product\View;

use Magento\Framework\Data\Collection;
use MagicToolbox\MagicThumb\Helper\Data;

class Gallery extends \Magento\Catalog\Block\Product\View\Gallery
{
    /**
     * Helper
     *
     * @var \MagicToolbox\MagicThumb\Helper\Data
     */
    public $magicToolboxHelper = null;

    /**
     * MagicThumb module core class
     *
     * @var \MagicToolbox\MagicThumb\Classes\MagicThumbModuleCoreClass
     */
    public $toolObj = null;

    /**
     * Collection factory
     *
     * @var \Magento\Framework\Data\CollectionFactory
     */
    protected $collectionFactory = null;

    /**
     * Rendered gallery HTML
     *
     * @var array
     */
    protected $renderedGalleryHtml = [];

    /**
     * ID of the current product
     *
     * @var integer
     */
    protected $currentProductId = null;

    /**
     * Do reload product
     *
     * @var bool
     */
    protected $doReloadProduct = false;

    /**
     * Internal constructor, that is called from real constructor
     *
     * @return void
     */
    protected function _construct()
    {
        parent::_construct();

        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();

        $this->magicToolboxHelper = $objectManager->get(\MagicToolbox\MagicThumb\Helper\Data::class);
        $this->toolObj = $this->magicToolboxHelper->getToolObj();
        $this->collectionFactory = $objectManager->get(\Magento\Framework\Data\CollectionFactory::class);

        $version = $this->magicToolboxHelper->getMagentoVersion();
        if (version_compare($version, '2.2.5', '<')) {
            $this->doReloadProduct = true;
        }

        //NOTE: for versions 2.2.x (x >=9), 2.3.x (x >=2)
        if (class_exists('\Magento\Catalog\Block\Product\View\GalleryOptions')) {
            $galleryOptions = $objectManager->get(\Magento\Catalog\Block\Product\View\GalleryOptions::class);
            $this->setData('gallery_options', $galleryOptions);
        }

        //NOTE: for versions 2.3.x (x >=2)
        if (version_compare($version, '2.3.2', '>=')) {
            $imageHelper = $objectManager->get(\Magento\Catalog\Helper\Image::class);
            $this->setData('imageHelper', $imageHelper);
        }
    }

    /**
     * Retrieve collection of gallery images
     *
     * @param \Magento\Catalog\Model\Product $product
     * @return Magento\Framework\Data\Collection
     */
    public function getGalleryImagesCollection($product = null)
    {
        static $images = [];
        if (is_null($product)) {
            $product = $this->getProduct();
        }
        $id = $product->getId();
        if (!isset($images[$id])) {
            if ($this->doReloadProduct) {
                $productRepository = \Magento\Framework\App\ObjectManager::getInstance()->get(
                    \Magento\Catalog\Model\ProductRepository::class
                );
                $product = $productRepository->getById($product->getId());
            }

            $images[$id] = $product->getMediaGalleryImages();
            if ($images[$id] instanceof \Magento\Framework\Data\Collection) {
                $baseMediaUrl = $this->_storeManager->getStore()->getBaseUrl(\Magento\Framework\UrlInterface::URL_TYPE_MEDIA);
                $baseStaticUrl = $this->_storeManager->getStore()->getBaseUrl(\Magento\Framework\UrlInterface::URL_TYPE_STATIC);
                $makeSquareImages = $this->toolObj->params->checkValue('square-images', 'Yes');

                //NOTE: to sort by position for associated products
                $collection = $this->collectionFactory->create();
                $iterator = $images[$id]->getIterator();
                $iterator->uasort(function ($a, $b) {
                    $aPos = (int)$a->getPosition();
                    $bPos = (int)$b->getPosition();
                    if ($aPos > $bPos) {
                        return 1;
                    } elseif ($aPos < $bPos) {
                        return -1;
                    }
                    return 0;
                });
                $iterator->rewind();
                while ($iterator->valid()) {
                    $collection->addItem($iterator->current());
                    $iterator->next();
                }
                $images[$id] = $collection;

                foreach ($images[$id] as $image) {
                    /* @var \Magento\Framework\DataObject $image */

                    $mediaType = $image->getMediaType();
                    if ($mediaType != 'image' && $mediaType != 'external-video') {
                        continue;
                    }

                    $img = $this->_imageHelper
                        ->init($product, 'product_page_image_large', ['width' => null, 'height' => null])
                        ->setImageFile($image->getFile())
                        ->getUrl();

                    $iPath = $image->getPath();
                    if (!is_file($iPath)) {
                        if (strpos($img, $baseMediaUrl) === 0) {
                            $iPath = str_replace($baseMediaUrl, '', $img);
                            $iPath = $this->magicToolboxHelper->getMediaDirectory()->getAbsolutePath($iPath);
                        } else {
                            $iPath = str_replace($baseStaticUrl, '', $img);
                            $iPath = $this->magicToolboxHelper->getStaticDirectory()->getAbsolutePath($iPath);
                        }
                    }
                    try {
                        $originalSizeArray = getimagesize($iPath);
                    } catch (\Exception $exception) {
                        $originalSizeArray = [0, 0];
                    }

                    if ($mediaType == 'image') {
                        if ($makeSquareImages) {
                            $bigImageSize = ($originalSizeArray[0] > $originalSizeArray[1]) ? $originalSizeArray[0] : $originalSizeArray[1];
                            $img = $this->_imageHelper
                                ->init($product, 'product_page_image_large')
                                ->setImageFile($image->getFile())
                                ->keepFrame(true)
                                ->resize($bigImageSize)
                                ->getUrl();
                        }
                        $image->setData('large_image_url', $img);

                        list($w, $h) = $this->magicToolboxHelper->magicToolboxGetSizes('thumb', $originalSizeArray);

                        $this->_imageHelper
                            ->init($product, 'product_page_image_medium', ['width' => $w, 'height' => $h])
                            ->setImageFile($image->getFile());
                        if ($makeSquareImages) {
                            $this->_imageHelper->keepFrame(true);
                        }
                        $medium = $this->_imageHelper->getUrl();
                        $image->setData('medium_image_url', $medium);
                        $image->setData('medium_image_width', $w);
                        $image->setData('medium_image_height', $h);                        
                    }

                    list($w, $h) = $this->magicToolboxHelper->magicToolboxGetSizes('selector', $originalSizeArray);

                    $this->_imageHelper
                        ->init($product, 'product_page_image_small', ['width' => $w, 'height' => $h])
                        ->setImageFile($image->getFile());
                    if ($makeSquareImages) {
                        $this->_imageHelper->keepFrame(true);
                    }
                    $thumb = $this->_imageHelper->getUrl();
                    $image->setData('small_image_url', $thumb);
                    $image->setData('small_image_width', $w);
                    $image->setData('small_image_height', $h);

                    if ($mediaType == 'external-video') {
                        $videoMetadata = $image->getVideoMetadata();
                        if (empty($videoMetadata)) {
                            $videoMetadata = $this->magicToolboxHelper->updateVideoMetadata(
                                $image->getValueId(),
                                $image->getVideoUrl(),
                                []
                            );
                        } else {
                            $videoMetadata = json_decode($videoMetadata, true);
                        }
                        if (is_array($videoMetadata)) {
                            if (isset($videoMetadata['width'], $videoMetadata['height'])) {
                                $image->setData('video_size', [
                                    'width' => $videoMetadata['width'],
                                    'height' => $videoMetadata['height']
                                ]);
                            }
                        }
                    }
                }
            }
        }

        return $images[$id];
    }

    /**
     * Retrieve original gallery block
     *
     * @return mixed
     */
    public function getOriginalBlock()
    {
        $data = $this->_coreRegistry->registry('magictoolbox');
        return is_null($data) ? null : $data['blocks']['product.info.media.image'];
    }

    /**
     * Retrieve another gallery block
     *
     * @return mixed
     */
    public function getAnotherBlock()
    {
        $data = $this->_coreRegistry->registry('magictoolbox');
        if ($data) {
            $skip = true;
            foreach ($data['blocks'] as $name => $block) {
                if ($name == 'product.info.media.magicthumb') {
                    $skip = false;
                    continue;
                }
                if ($skip) {
                    continue;
                }
                if ($block) {
                    return $block;
                }
            }
        }
        return null;
    }

    /**
     * Check for installed modules, which can operate in cooperative mode
     *
     * @return bool
     */
    public function isCooperativeModeAllowed()
    {
        $data = $this->_coreRegistry->registry('magictoolbox');
        return is_null($data) ? false : $data['cooperative-mode'];
    }

    /**
     * Get thumb switcher initialization attribute
     *
     * @param integer $id
     * @return string
     */
    public function getThumbSwitcherInitAttribute($id = null)
    {
        static $html = null;
        if ($html === null) {
            if (is_null($id)) {
                $id = $this->currentProductId;
            }
            $settings = $this->magicToolboxHelper->getVideoSettings();
            $settings['tool'] = 'magicthumb';
            $settings['switchMethod'] = $this->toolObj->params->getValue('selectorTrigger');
            if ($settings['switchMethod'] == 'hover') {
                $settings['switchMethod'] = 'mouseover';
            }
            $settings['productId'] = $id;
            $html = ' data-mage-init=\'{"magicToolboxThumbSwitcher": '.json_encode($settings).'}\'';
        }
        return $html;
    }

    /**
     * Before rendering html, but after trying to load cache
     *
     * @return $this
     */
    protected function _beforeToHtml()
    {
        $this->renderGalleryHtml();
        return parent::_beforeToHtml();
    }

    /**
     * Get rendered HTML
     *
     * @param integer $id
     * @return string
     */
    public function getRenderedHtml($id = null)
    {
        if (is_null($id)) {
            $id = $this->getProduct()->getId();
        }
        return isset($this->renderedGalleryHtml[$id]) ? $this->renderedGalleryHtml[$id] : '';
    }

    /**
     * Render gallery block HTML
     *
     * @param \Magento\Catalog\Model\Product $product
     * @param bool $isAssociatedProduct
     * @param array $data
     * @return $this
     */
    public function renderGalleryHtml($product = null, $isAssociatedProduct = false, $data = [])
    {
        if (is_null($product)) {
            $product = $this->getProduct();
        }
        $this->currentProductId = $id = $product->getId();
        if (!isset($this->renderedGalleryHtml[$id])) {
            $this->toolObj->params->setProfile('product');
            $name = $product->getName();
            $productImage = $product->getImage();
            $description = $product->getDescription();
            $mainHTML = '';
            $defaultContainerId = 'mtImageContainer';
            $containersData = [
                'mtImageContainer' => '',
                'mt360Container' => '',
                'mtVideoContainer' => '',
            ];
            $selectorsArray = [];

            $images = $this->getGalleryImagesCollection($product);

            $originalBlock = $this->getOriginalBlock();

            if (!$images->count()) {
                $this->renderedGalleryHtml[$id] = $isAssociatedProduct ? '' : $this->getPlaceholderHtml();
                return $this;
            }

            $selectorIndex = 0;
            $baseIndex = 0;
            foreach ($images as $image) {

                $mediaType = $image->getMediaType();
                $isImage = $mediaType == 'image';
                $isVideo = $mediaType == 'external-video';

                if (!$isImage && !$isVideo) {
                    continue;
                }

                $label = $isImage ? $image->getLabel() : $image->getVideoTitle();
                if (empty($label)) {
                    $label = $name;
                }

                if ($isImage) {
                    if (empty($containersData['mtImageContainer']) || $productImage == $image->getFile()) {
                        $containersData['mtImageContainer'] = $this->toolObj->getMainTemplate([
                            'id' => '-product-'.$id,
                            'img' => $image->getData('large_image_url'),
                            'thumb' => $image->getData('medium_image_url'),
                            'title' => $label,
                            'alt' => $label,
                            'width' => $image->getData('medium_image_width'),
                            'height' => $image->getData('medium_image_height'),
                            'description' => $description,
                        ]);
                        $containersData['mtImageContainer'] = '<div>'.$containersData['mtImageContainer'].'</div>';
                        if ($selectorIndex == 0 || $productImage == $image->getFile()) {
                            $defaultContainerId = 'mtImageContainer';
                            $containersData['mtVideoContainer'] = '';
                            $baseIndex = $selectorIndex;
                        }
                    }
                    $selectorsArray[] = $this->toolObj->getSelectorTemplate([
                        'id' => '-product-'.$id,
                        'group' => 'product-page',
                        'img' => $image->getData('large_image_url'),
                        'thumb' => $image->getData('small_image_url'),
                        'medium' => $image->getData('medium_image_url'),
                        'title' => $label,
                        'alt' => $label,
                        'width' => $image->getData('small_image_width'),
                        'height' => $image->getData('small_image_height'),
                    ]);
                } else {
                    $videoSize = $image->getData('video_size');
                    if ($videoSize) {
                        $videoSize = ' data-video-size=\'' . json_encode($videoSize) . '\'';
                    } else {
                        $videoSize = '';
                    }
                    if ($selectorIndex == 0 || $productImage == $image->getFile()) {
                        $defaultContainerId = 'mtVideoContainer';
                        $containersData['mtVideoContainer'] = '<div class="product-video init-video" data-video="' . $image->getVideoUrl() . '"' . $videoSize . '></div>';
                        $baseIndex = $selectorIndex;
                    }

                    $videoUrl = preg_replace('/https?:\/\/youtu\.be\/(.*?)$/ims', 'https://www.youtube.com/watch?v=$1', $image->getVideoUrl());
                    $selectorsArray[] =
                        '<a class="video-selector" data-thumb-id="MagicThumbImage-product-' . $id . '" href="' . $videoUrl . '"' . $videoSize . ' title="' . $label . '">' .
                        '<img src="' . $image->getData('small_image_url') . '" alt="' . $label . '" />' .
                        '</a>';
                }

                $selectorIndex++;
            }

            //NOTE: cooperative mode
            if (isset($data['magic360-html'])) {
                if ($data['magic360-position'] == 0 || empty($selectorsArray)) {
                    $defaultContainerId = 'mt360Container';
                    $containersData['mtVideoContainer'] = '';
                }
                $containersData['mt360Container'] = $data['magic360-html'];
                if (isset($data['magic360-icon'])) {
                    $data['magic360-icon'] =
                        '<a class="m360-selector" title="360" href="#" onclick="return false;">'.
                        '<img class="" src="'.$data['magic360-icon'].'" alt="360" />'.
                        '</a>';
                    $selectorsArray = array_merge(
                        array_slice($selectorsArray, 0, $data['magic360-position']),
                        [$data['magic360-icon']],
                        array_slice($selectorsArray, $data['magic360-position'])
                    );
                    if ($defaultContainerId == 'mt360Container') {
                        $baseIndex = 0;
                    } elseif ($baseIndex >= $data['magic360-position']) {
                        $baseIndex++;
                    }
                }
            }

            foreach ($selectorsArray as $i => &$selector) {
                $class = 'mt-thumb-switcher '.($i == $baseIndex ? 'active-selector ' : '');
                if (preg_match('#(<a(?=\s)[^>]*?(?<=\s)class=")([^"]*+")#i', $selector, $match)) {
                    $selector = str_replace($match[0], $match[1].$class.$match[2], $selector);
                } else {
                    $selector = str_replace('<a ', '<a class="'.$class.'" ', $selector);
                }
            }

            foreach ($containersData as $containerId => $containerHTML) {
                $displayStyle = $defaultContainerId == $containerId ? 'block' : 'none';
                $mainHTML .= "<div id=\"{$containerId}\" style=\"display: {$displayStyle};\">{$containerHTML}</div>";
            }

            if (empty($selectorsArray)) {
                if ($originalBlock) {
                    $this->renderedGalleryHtml[$id] = $isAssociatedProduct ? '' : $this->getPlaceholderHtml();
                }
                return $this;
            }
            $additionalClasses = '';
            $scrollOptions = '';
            if ($scroll = $this->magicToolboxHelper->getScrollObj()) {
                $additionalClasses = $this->toolObj->params->getValue('scroll-extra-styles');
                if (empty($additionalClasses)) {
                    $additionalClasses = 'MagicScroll';
                } else {
                    $additionalClasses = 'MagicScroll '.trim($additionalClasses);
                }

                //NOTE: temporary fix because of issue with loading icon
                $scroll->params->setValue('lazy-load', 'No', 'magicthumb-magicscroll-product');

                $scrollOptions = $scroll->params->serialize(false, '', 'magicthumb-magicscroll-product');

                //NOTE: disable MagicScroll on page load to start manually
                $scrollOptions = 'autostart:false;'.$scrollOptions;

                if (!empty($scrollOptions)) {
                    $scrollOptions = " data-options=\"{$scrollOptions}\"";
                }
            }
            $selectorMaxWidth = (int)$this->toolObj->params->getValue('selector-max-width');
            $thumbSwitcherOptions = '';
            if (!$isAssociatedProduct) {
                $thumbSwitcherOptions = $this->getThumbSwitcherInitAttribute();
            }

            $layout = $this->toolObj->params->getValue('template');
            $templateData = [
                'layout' => $layout,
                'id' => $id,
                'mainHTML' => $mainHTML,
                'selectorsArray' => $selectorsArray,
                'selectorMaxWidth' => $selectorMaxWidth,
                'additionalClasses' => $additionalClasses,
                'scrollOptions' => $scrollOptions,
                'thumbSwitcherOptions' => $thumbSwitcherOptions,
            ];
            $galleryBlock = $this;
            if (!$this->getParentBlock()) {
                $data = $this->_coreRegistry->registry('magictoolbox');
                $galleryBlock = $data['blocks'][$data['current']];
            }
            $templateBlock = $galleryBlock->addChild(
                'layout' . $id,
                \MagicToolbox\MagicThumb\Block\Product\View\Layouts::class,
                $templateData
            );
            $this->renderedGalleryHtml[$id] = $templateBlock->toHtml();
        }
        return $this;
    }

    /**
     * Get placeholder HTML
     *
     * @return string
     */
    public function getPlaceholderHtml()
    {
        static $html = null;
        if ($html === null) {
            $placeholderUrl = $this->_imageHelper->getDefaultPlaceholderUrl('image');
            list($width, $height) = $this->magicToolboxHelper->magicToolboxGetSizes('thumb');
            $html = '<div class="MagicToolboxContainer placeholder"'.$this->getThumbSwitcherInitAttribute().' style="width: '.$width.'px;height: '.$height.'px">'.
                    '<span class="align-helper"></span>'.
                    '<img src="'.$placeholderUrl.'"/>'.
                    '</div>';
        }
        return $html;
    }
}
